/* FILE: mlatlon.c                              (D. Tottingham  09/01/89)

This is a collection of C helper functions that manage latlon coordinate
transformations for xdetect.  All functions have been written and compiled
medium model.  The following functions are included:

ll_fold ()                      convert lat, lon to geocentric lat, lon
ll_to_latlon ()                 convert delta lat, lon to lat, lon
ll_to_xy ()                     convert lat, lon to x, y

EXTERNAL FUNCTIONS CALLED:
   none

HISTORY:
   none

*/


/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <math.h>

#include "mconst.h"
#include "mlatlon.h"


/*=======================================================================*
 *                                hypot                                  *
 *=======================================================================*/
/* Calculate euclidian distance accurately without overflow.             */

PRIVATE
double hypot (x, y)
double x, y;
{
   double new_x, new_y, temp;

   new_x = fabs(x);
   new_y = fabs(y);

   /* Swap new_x and new_y if new_x > new_y */
   if (new_x > new_y) {
      temp = new_x; new_x = new_y; new_y = temp;
   }

   if (new_y) {
      new_x /= new_y;
      return ( new_y * sqrt (new_x * new_x + 1.0));
   } else return (0.0);
}

/*=======================================================================*
 *                             xy_to_polar                               *
 *=======================================================================*/
/* Convert x, y to r, theta                                              */

PRIVATE
LL_POLAR xy_to_polar (x, y)
double x, y;
{
   LL_POLAR p;

   p.r = hypot (x, y);
   p.theta = 0.0;
   if (y || x) p.theta = atan2 (y, x);

   return (p);
}

/*=======================================================================*
 *                          latlon_to_polar                              *
 *=======================================================================*/
/* Convert geocentric lat, lon to polar coordinates.                     */

PRIVATE
LL_POLAR latlon_to_polar (latlon, mean_latlon)
LL_LATLON latlon, mean_latlon;
{
   LL_POLAR p1;
   double st0, st1, ct0, ct1, sdlon, cdlon, cdelta, radius, t1, t2;

   st0 = cos(mean_latlon.lat);
   ct0 = sin(mean_latlon.lat);
   t1 = (st0 * st0) / EQURAD / EQURAD;
   t2 = (ct0 * ct0) / POLRAD / POLRAD;
   radius = 1.0 / sqrt (t1 + t2);
   st1 = cos(latlon.lat);
   ct1 = sin(latlon.lat);
   sdlon = sin(mean_latlon.lon - latlon.lon);
   cdlon = cos(mean_latlon.lon - latlon.lon);
   cdelta = (st0 * st1 * cdlon) + (ct0 * ct1);

   p1 = xy_to_polar (((st0 * ct1) - (st1 * ct0 * cdlon)), st1 * sdlon);
   p1.r = radius * atan2 (p1.r, cdelta);

   if (p1.theta < 0.0) p1.theta += TWOPI;
   else if (p1.theta >= TWOPI) p1.theta -= TWOPI;
   p1.theta *= DEG;

   return (p1);
}

/*=======================================================================*
 *                               unfold                                  *
 *=======================================================================*/
/* Convert geocentric lat, lon to lat, lon.                              */

PRIVATE
LL_LATLON unfold (lat, lon)
double lat, lon;
{
   LL_LATLON latlon;

   if ((HALFPI - fabs(lat)) >= 0.02)
      latlon.lat = C1 * (lat + ((lat >= 0) ? C2 : -C2));
   else latlon.lat = atan ( tan(lat) / C1);

   latlon.lat *= DEG;
   latlon.lon = lon * DEG;

   return (latlon);
}

/*=======================================================================*
 *                              ll_fold                                  *
 *=======================================================================*/
/* Convert lat, lon to geocentric lat, lon.                              */

PUBLIC
LL_LATLON ll_fold (lat, lon)
double lat, lon;
{
   LL_LATLON geocentric;
   int sign;

   /* Compute the geocentric latitude */
   sign = (lat >= 0) ? 1 : -1;
   geocentric.lat = fabs(lat) * RAD;
   if ((HALFPI - geocentric.lat) >= 0.02)
      geocentric.lat = atan (C1 * tan(geocentric.lat));
   else geocentric.lat /= C1 - C2;
   geocentric.lat *= sign;

   /* Compute the geocentric longitude */
   geocentric.lon = lon * RAD;

   return (geocentric);
}

/*=======================================================================*
 *                              ll_to_latlon                             *
 *=======================================================================*/
/* Convert delta lat, lon to lat, lon.  The parameter pair mean_lat,
   mean_long is the latitude, longitude reference coordinate where
   x,y = (0, 0).                                                         */

PUBLIC
LL_LATLON ll_to_latlon (delta_lat, delta_lon, mean_latlon)
double delta_lat, delta_lon;
LL_LATLON mean_latlon;
{
   LL_LATLON latlon;
   LL_POLAR  p1, p2;
   double st0, st1, ct0, ct1, cz0, radius, sdelta, cdelta, t1, t2;
   double lat, lon;

   p1 = xy_to_polar (delta_lat, delta_lon);
   if (p1.theta < 0.0) p1.theta += TWOPI;

   st0 = cos (mean_latlon.lat); ct0 = sin (mean_latlon.lat);

   t1 = (st0 * st0) / EQURAD / EQURAD;
   t2 = (ct0 * ct0) / POLRAD / POLRAD;
   radius = 1.0 / sqrt (t1 + t2);
   sdelta = sin (p1.r / radius);
   cdelta = cos (p1.r / radius);
   cz0 = cos (p1.theta);
   ct1 = (st0 * sdelta * cz0) + (ct0 * cdelta);

   p2 = xy_to_polar (((st0 * cdelta) - (ct0 * sdelta * cz0)), sdelta* sin(p1.theta));
   lat = atan2 (ct1, p2.r);
   lon = mean_latlon.lon + p2.theta;

   if (fabs(lon) > PI) lon -= (lon >= 0) ? TWOPI : -TWOPI;
   latlon = unfold (lat, lon);
   return (latlon);
}

/*=======================================================================*
 *                              ll_to_xy                                 *
 *=======================================================================*/
/* Convert lat, lon to x, y.  The parameter pair mean_lat, mean_long is
   the latitude, longitude reference coordinate where x,y = (0, 0).      */

PUBLIC
LL_XY ll_to_xy (lat, lon, mean_latlon)
double lat, lon;
LL_LATLON mean_latlon;
{
   LL_POLAR p;
   LL_XY rect;

   p = latlon_to_polar (ll_fold (lat, lon), mean_latlon);
   rect.x = p.r * sin (p.theta * RAD);
   rect.y = p.r * cos (p.theta * RAD);
   return (rect);
}
